using System;
using System.Xml;
using System.Collections;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// This class represents array RPC parameters used in RPC request.	
	/// </summary>
	public class RpcArrayParameter : RpcParameter
	{
		private static readonly Type _baseType = typeof(Hashtable);

		// Constants used in XML serialization/deserialization
		private const string XMLCONSTS_PARAMETER_TYPE_NAME = "array";
		private const string XMLCONSTS_INDEX_NODE_NAME = "Index";
		private const string XMLCONSTS_INDICES_NODE_NAME = "Indices";
		private const string XMLCONSTS_NAME_ATTRIBUTE_NAME = "name";
		private const string XMLCONSTS_VALUE_ATTRIBUTE_NAME = "value";

		/// <summary>
		/// Static constructor getting rid of 'beforefieldinit' attribute.
		/// </summary>
		static RpcArrayParameter() {}

		/// <summary>
		/// Shortcut constructor that does not initialize parameter value.
		/// </summary>
		/// <param name="paramPosition">RPC parameter position in remote procedure's signature.</param>
		public RpcArrayParameter( int paramPosition )
			: this( paramPosition, null ) {}

		/// <summary>
		/// Constructor initializing both parameter name and value.
		/// </summary>
		/// <param name="paramPosition">RPC parameter position in remote procedure's signature.</param>
		/// <param name="paramValue">RPC parameter value.</param>
		public RpcArrayParameter( int paramPosition, Hashtable paramValue )
			: base( XMLCONSTS_PARAMETER_TYPE_NAME, new ParameterPosition( paramPosition ), paramValue ) {}

		/// <summary>
		/// XML deserialization constructor creating parameter object from parameter XML element.
		/// </summary>
		/// <param name="parameterNode">XML element to parse and get parameter information from.</param>
		public RpcArrayParameter( XmlNode parameterNode )
			: base( XMLCONSTS_PARAMETER_TYPE_NAME, parameterNode ) {}

		/// <summary>
		/// Xml serialization method writing out RPC parameter value.
		/// </summary>
		/// <param name="writer">XmlWriter to use.</param>
		protected override void WriteParameterValue( XmlWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			writer.WriteStartElement( XMLCONSTS_INDICES_NODE_NAME );

			Hashtable _bufHt = (Hashtable)BaseValue;
					
			foreach( string _indexName in _bufHt.Keys )
			{
				writer.WriteStartElement( XMLCONSTS_INDEX_NODE_NAME );
				writer.WriteAttributeString( XMLCONSTS_NAME_ATTRIBUTE_NAME, _indexName );
				writer.WriteAttributeString( XMLCONSTS_VALUE_ATTRIBUTE_NAME, (string)_bufHt[_indexName] );
				writer.WriteEndElement();
			}

			writer.WriteEndElement();
		}

		/// <summary>
		/// XML deserialization method reading parameter value.
		/// </summary>
		/// <param name="parameterElement">
		///		Parameter element. Parameter value resides in a child node of this element.
		///	</param>
		protected override object ParseParameterValue( XmlElement parameterElement )
		{
			if( parameterElement == null )
				throw( new ArgumentNullException( "parameterElement" ) );

			XmlElement _indicesRootElement = XmlUtility.ParseGetRequiredElementByUniqueTagName( parameterElement, XMLCONSTS_INDICES_NODE_NAME );

			Hashtable _result = new Hashtable();

			foreach( XmlElement _arrayEntryElement in _indicesRootElement.GetElementsByTagName( XMLCONSTS_INDEX_NODE_NAME ) )
			{
				string _indexKey = XmlUtility.ParseGetRequiredAttributeValue( _arrayEntryElement, XMLCONSTS_NAME_ATTRIBUTE_NAME );
				
				if( _result.ContainsKey( _indexKey ) )
					throw( new XmlParseException( SR.Exceptions.DuplicateIndexKeyDuringRpcArrayParameterDeserialization( _indexKey ) ) );

				_result.Add( _indexKey, XmlUtility.ParseGetRequiredAttributeValue( _arrayEntryElement, XMLCONSTS_VALUE_ATTRIBUTE_NAME ) );
			}

			return _result;
		}

		/// <summary>
		/// Conversion operator overload added for ease of use. 
		/// </summary>
		/// <param name="src">Source RPC parameter.</param>
		/// <returns>Parameter value.</returns>
		public static implicit operator Hashtable( RpcArrayParameter src )
		{
			return (Hashtable)src.BaseValue;
		}

		/// <summary>
		/// This method is used during XML deserialization to recognize parameter XML.
		/// It tests supplied string for match with parameter type name and 
		/// returns true when there is a match.
		/// </summary>
		/// <param name="parameterTypeName">String to tests (expected to be a parameter type name).</param>
		/// <returns>True if supplied string matches parameter type name.</returns>
		public static bool DoesParameterTypeNameMatch( string parameterTypeName )
		{
			return parameterTypeName == XMLCONSTS_PARAMETER_TYPE_NAME;
		}

		/// <summary>
		/// RPC parameter value.
		/// </summary>
		public Hashtable Value
		{
			get
			{
				return (Hashtable)BaseValue;
			}
			set
			{
				SetBaseValue( value );
			}
		}

		/// <summary>
		/// RPC parameter's underlying .NET type. 
		/// </summary>
		public override Type BaseType
		{
			get
			{
				return _baseType;
			}
		}
	}
}
